/*
 * Routines for resolving xbars whose physical location in the fabric is
 * unknown.
 *
 * When the fabric contains xbars without xbar IDs (16 port xbars), the xbars
 * are initially only locatable relative to other xbars and hosts. I.e. their
 * actual phyiscal location in an enclosure and/or linecard are not known. In
 * order to determine their physical location, invalid packets are sent to xbars
 * and each port in each xbar in each linecard in each enclosure is inspected
 * via the monitoring line cards looking for a jump in invalid packet counts.
 * Of course, this will only work if there are monitoring line cards in all
 * enclosures and fms has been told about them via fm_switch.
 *
 * When a topo map is sent in from and FMA any xbars that cannot be associated
 * with xbars in the fabric are assigned their own "bogus" enclosure and 
 * linecards and inserted into fabric. "Resolving" xbars means to try to 
 * find the "real" xbars corresponding to each of the "bogus" xbars, copying
 * over the links and then removing the bogus enclosure/linecard/xbar. 
 *
 * Since the internal connections within a real enclosure are well defined,
 * once a single xbar in an enclousre is resolved, all of the xbars in the 
 * enclosure should be able to be resolved. The is done by following the 
 * links of the real xbars in a depth-first recursive manner. When there are
 * no more ports to explore and the corresponding bogus xbar links have all
 * been copied over, the bogus xbar can be eliminated.
 */

#include <sys/types.h>
#include <netinet/in.h>

#include "db.h"
#include "libfma.h"
#include "lf_fabric.h"
#include "lf_load_fabric.h"
#include "lf_topo_map.h"
#include "lf_scheduler.h"
#include "lf_fms_comm.h"
#include "lf_channel.h"
#include "lf_switch.h"


#include "fms.h"
#include "fms_error.h"
#include "fms_fabric.h"
#include "fms_fabric_delta.h"
#include "fms_fma.h"
#include "fms_notify.h"
#include "fms_resolve.h"
#include "fms_state.h"
#include "fms_switch.h"
#include "fms_settings.h"

/*
 * Local prototypes
 */
static void fms_res_save_inv_rt_counts(void);
static void fms_res_calc_inv_rt_diffs(void);
static void fms_find_xbar();
static void fms_send_inv_rts();
static int fms_count_bogus_xbars();
static struct lf_xbar *fms_get_bogus_xbar();
static struct lf_xbar *fms_find_xbar_with_inv_rts(int *port);
static void bond_xbars( struct lf_xbar *rxp, struct lf_xbar *bxp,
    int bp_in, int rp_in);

/*
 * Check if xbars need resolution
 */
int
fms_fabric_needs_resolution()
{
  int x;
  struct lf_fabric *fp;
  int num_bogus_xbars;
  int num_unresolved_xbars;

  /* get the current fabric pointer */
  fp = F.fabvars->fabric;

  /* rebuild fp->xbars in case prior run of fms_resolve_bogus_xbars_start
   * rearranged or removed some xbars */
  lf_build_xbar_array(fp);

  /* Count "real" xbars that are not resolved */
  num_unresolved_xbars = 0;
  for (x=0; x<fp->num_xbars; ++x) {
    struct lf_xbar *xp;
    xp = fp->xbars[x];
    if (! FMS_ENC(xp->linecard->enclosure)->is_bogus
        && ! FMS_XBAR(xp)->mrf_resolved) {
      num_unresolved_xbars++;
    }
  }

  /* get the current number of bogus xbars */
  num_bogus_xbars = fms_count_bogus_xbars();

  /* If there are no bogus xbars or if there are no more
   * unresolved xbars there is nothing to do */
  if (num_bogus_xbars == 0 || num_unresolved_xbars == 0) {
    fms_notify(FMS_EVENT_DEBUG, "No unresolved xbars");
    return FALSE;
  }

  /* Check if all xbars are bogus */
  if (fp->num_enclosures == num_bogus_xbars) {
    fms_notify(FMS_EVENT_ERROR, "%s\n%s",
        "No valid enclosures in db- complete xbar resolution not possible.",
        "Add enclosures with \"fm_switch\"");
    return FALSE;
  }

  /*
   * If there are unresolved xbars and there are more unexplored xbars 
   * than there used to be, there is work to do.
   */
  if (num_bogus_xbars > FMS_FABRIC(fp)->num_bogus_xbars) {
    fms_notify(FMS_EVENT_DEBUG, "%d xbars need resolving",
        num_bogus_xbars - FMS_FABRIC(fp)->num_bogus_xbars);
    return TRUE;
  } else {
    /* There are unresolved xbars but we've already explored all the current
     * set of bogus xbars and have not been able to resolve them. So at this
     * point * there no more resolution can be done (maybe some more linecards
     * will come online later).
     */
    fms_notify(FMS_EVENT_DEBUG, "Unresolved xbars can't currently be resolved");
    return FALSE;
  }
}

/*
 * resolve bogus xbars using invalid routes
 */
void
fms_resolve_fabric()
{
  struct lf_fabric *fp;

  /* get the current fabric pointer */
  fp = F.fabvars->fabric;

  if (FMS_FABRIC(fp)->resolve_in_prog == TRUE) {
    fms_notify(FMS_EVENT_INFO,
      "Fabric resolution already in progress- aborting this attempt");
    return;
  } else {
    FMS_FABRIC(fp)->resolve_in_prog = TRUE;
  }

  /* reset counters */
  FMS_FABRIC(fp)->inv_rt_retries = 0;

  /* kick off xbar resolution by setting a callback to send invalid
   * routes after getting a snapshot of the invalid route counts */
  fms_switch_set_res_handler(fms_send_inv_rts);

  return;
}

/*
 * Abort resolution
 */
void
fms_stop_fabric_resolution()
{
  struct lf_fabric *fp;

  /* get the current fabric pointer */
  fp = F.fabvars->fabric;

  fms_notify(FMS_EVENT_INFO, "Aborting fabric resolution");

  FMS_FABRIC(fp)->resolve_in_prog = FALSE;
  fms_switch_clear_res_handler();
}

/*
 * Save the current set of invalid route counts
 */
static void
fms_res_save_inv_rt_counts()
{
  int x;
  struct lf_fabric *fp;

  fp = F.fabvars->fabric;

  for (x=0; x<fp->num_xbars; ++x) {
    struct lf_xbar *xp;
    struct lf_xbarport_data *dp;
    int p;

    xp = fp->xbars[x];
    dp = xp->data;

    /* "bogus" xbars will have no dp */
    if (dp == NULL) continue;

    for (p=0; p<xp->num_ports; ++p) {
      struct fms_xbar_port *pp;

      pp = FMS_XBAR(xp)->x_port_data[p];

      /* Save this xbar's current invalid route count */
      pp->res_inv_rt_count = dp->cur_vals.invalidroutes;

      ++dp;	/* next data set */
    }
  }
}

/*
 * Calculate invalid route diffs
 */
static void
fms_res_calc_inv_rt_diffs()
{
  int x;
  struct lf_fabric *fp;

  fp = F.fabvars->fabric;

  for (x=0; x<fp->num_xbars; ++x) {
    struct lf_xbar *xp;
    struct lf_xbarport_data *dp;
    int p;

    xp = fp->xbars[x];

    /* skip ethernet cards! */
    if (strcmp(xp->linecard->product_id, "M3-SW16-8E") == 0) {
      continue;
    }

    dp = xp->data;

    /* "bogus" xbars will have no dp */
    if (dp == NULL) continue;

    for (p=0; p<xp->num_ports; ++p) {
      struct fms_xbar_port *pp;

      pp = FMS_XBAR(xp)->x_port_data[p];

      /* Calculate # of invalid route packets received */
      pp->res_inv_rt_diff = dp->cur_vals.invalidroutes - pp->res_inv_rt_count;

      ++dp;	/* next data set */
    }
  }
}


/*
 * A snapshot of invalid route counts has just completed, save the
 * values and send invalid route packets for next bogus xbar 
 */
static void
fms_send_inv_rts()
{

  struct lf_fma_send_inv_rt msg;
  struct lf_fabric *fp;
  struct lf_xbar *xp;
  struct fms_fma_desc *adp;
  int h;
  int rc;

  fp = F.fabvars->fabric;

  /* Get the next bogus xbar (if any) */
  xp = fms_get_bogus_xbar();

  if (xp == NULL) LF_ERROR(("No bogus xbars"));

  /* Save current invalid route values */
  fms_res_save_inv_rt_counts();

  /* find the first FMA in host list */
  adp = NULL;
  for (h=0; h<fp->num_hosts; ++h) {
    adp = FMS_HOST(fp->hosts[h])->adp;
    if (adp != NULL) break;
  }

  /* If no adp found, then no one to send invalid routes */
  if (adp == NULL) {
    fms_notify(FMS_EVENT_INFO, "No FMAs from which to send invalid routes");
    return;
  }

  /* Initiate sending invalid routes to unknown xbar from FMA */
  FMS_FABRIC(fp)->inv_rt_seq_no++;
  msg.ir_seq_no_32 = htonl(FMS_FABRIC(fp)->inv_rt_seq_no);
  msg.ir_xbar_index_32 = htonl(xp->x_topo_index);
  msg.ir_num_pkts_32 = htonl(F.settings->resolve_packet_send_count);
  rc = fms_fma_write(adp, LF_FMA_SEND_INVALID_ROUTE,
		     &msg, sizeof(msg));
  if (rc == -1) {
    LF_ERROR(("Error sending xbar invalid routes request to FMA"));
  }

  fms_notify(FMS_EVENT_DEBUG,
      "Invalid route req for %s (xb %d, seq:%d) sent to %s",
      xp->linecard->enclosure->name, xp->x_topo_index,
      FMS_FABRIC(fp)->inv_rt_seq_no, adp->hostname);

  FMS_FABRIC(fp)->inv_rt_retries++;

  return;

 except:
  fms_perror_exit(1);
}

/*
 * Schedule a switch examination in FMS_SWITCH_INSPECTION_DELAY_MS. The delay
 * gives the switch time to update it's counters. This function is a re-entry
 * point into the main thread resolve code, so need to check if resolving got
 * cancelled.
 */
void
fms_send_inv_rts_complete(
  struct fms_fma_desc *adp,
  struct fma_fms_inv_rt_msg *inv_rt_p)
{
  struct lf_fabric *fp;
  struct fms_settings *fsetp;
  int seq_no;
  int host_in_port;
  int send_rslt;

  fp = F.fabvars->fabric;
  fsetp = F.settings;

  send_rslt = ntohl(inv_rt_p->irr_send_succeeded_32);
  seq_no = ntohl(inv_rt_p->irr_seq_no_32);
  host_in_port = ntohl(inv_rt_p->irr_host_in_port_32);

  fms_notify(FMS_EVENT_DEBUG, "inv rt complete received from %s, seq:%d,p:%d",
      adp->hostname, seq_no, host_in_port);

  if (FMS_FABRIC(fp)->resolve_in_prog == FALSE) {
    fms_notify(FMS_EVENT_DEBUG,
	"inv_rt sends complete, but fabric resolution aborted");
    return;
  }

  if (send_rslt == TRUE) {

    /* did we get the expected response packet? */
    if (FMS_FABRIC(fp)->inv_rt_seq_no != seq_no)
      fms_notify(FMS_EVENT_DEBUG,
          "Unexpected Invalid route message, exp: %d, received %d",
          FMS_FABRIC(fp)->inv_rt_seq_no, seq_no);
      
    /* save inv rt count results from FMA and schedule a count of inv rts */
    FMS_FABRIC(fp)->inv_rt_host_in_port = host_in_port;

    /* Set handler to call our invalid route inspector after next
     * set of queries finishes
     */
    fms_switch_set_res_handler(fms_find_xbar);

  } else {
    /* error sending invalid routes from FMA, retry */
    if (FMS_FABRIC(fp)->inv_rt_retries < fsetp->resolve_retries) {
      fms_notify(FMS_EVENT_DEBUG,
          "Invalid route send from %s failed. retrying", adp->hostname);

      fms_switch_set_res_handler(fms_send_inv_rts);

    /* No retries left, give up */
    } else {
      fms_notify(FMS_EVENT_ERROR,
          "Invalid route retries exceeded. Unable to resolve all xbars",
          adp->hostname);
      /* XXX still in progress ? */
    }
  }
}

/* 
 * Count the number of bogus xbars
 */
static int
fms_count_bogus_xbars()
{
  struct lf_fabric *fp;
  struct lf_enclosure *ep;
  int e;
  int num_bogus_xbars;

  /* get the current fabric pointer */
  fp = F.fabvars->fabric;

  /* assume there is only 1 bogus xbar per bogus enclosure */
  num_bogus_xbars = 0;
  for (e=0; e<fp->num_enclosures; ++e) {
    ep = fp->enclosures[e];
    if (FMS_ENC(ep)->is_bogus) num_bogus_xbars++;
  }
  return num_bogus_xbars;
}

/*
 * Return a bogus xbar or NULL if none. Try to select a different one each
 * time (this function is used to select the next xbar to send invalid routes
 * on and it is better to retry on a new xbars in case of failure rather than
 * retry the same one over and over).
 */
static struct lf_xbar *
fms_get_bogus_xbar()
{
  struct lf_fabric *fp;
  int e;
  struct lf_xbar *xp, *first_xp;

  first_xp = NULL;
  xp = NULL;

  fp = F.fabvars->fabric;

  for (e=0; e<fp->num_enclosures; ++e) {
    struct lf_enclosure *ep;
    int l;

    ep = fp->enclosures[e];
    if (FMS_ENC(ep)->is_bogus != TRUE) continue;

    for (l=0; l<ep->num_slots; ++l) {
      struct lf_linecard *lp;
      int x;

      lp = ep->slots[l];
      if (lp == NULL) continue;

      for (x=0; x<lp->num_xbars; ++x) {
        xp = LF_XBAR(lp->xbars[x]);
        if (xp != NULL) {
          if (first_xp == NULL) first_xp = xp;
          if (xp->x_topo_index > FMS_FABRIC(fp)->inv_rt_xbar_index) {
            FMS_FABRIC(fp)->inv_rt_xbar_index = xp->x_topo_index;
            return(xp);
          }
        }
      }
    }
  }

  if (first_xp != NULL) {
    FMS_FABRIC(fp)->inv_rt_xbar_index = first_xp->x_topo_index;
    return(first_xp);
  }

  return NULL ;
}

/*
 * Explore ports and copy links from bogus to real xbar.
 */
static void
bond_xbars(
  struct lf_xbar *rxp,   /* "real" (physical) node */
  struct lf_xbar *bxp,   /* "bogus" (topo) node */
  int bp_in,           
  int rp_in)          
{
  struct lf_xbar *obxp; /* bogus xbar's mate */
  struct lf_xbar *orxp; /* real xbar's mate */
  union lf_node *obnp; /* other bogus node ptr (the xbar bxp points to) */
  union lf_node *ornp; /* other real xbar ptr (the xbar rxp points to) */
  struct lf_xcvr *bxcp;  /* bogus xcvr */
  int obp;              /* other bogus node port that bxp connects to */
  int orp;              /* other real node port that rxp connects to */
  int bp;                /* port number on bogus xbar */ 
  int rp;                /* real xbar port */
  int dp;

  dp = bp_in - rp_in;

  /* topo xbars are in the 0-31 range and reals in the 0-15 range but the 
   * delta should never be > 15.
   */
  if (dp < -15 || dp >15) {
    LF_ERROR(("invalid bogus/real xbar port delta bp:%d, rp:%d, dp:%d",
        bp_in, rp_in, dp));
  }

  fms_notify(FMS_EVENT_DEBUG, "bond real:%d to bogus:%d, delta:%d",
      rxp->x_topo_index, bxp->x_topo_index, dp);

  /* mark both the real and bogus nodes as resolved ("resolved" for bogus
   * xbars really means more like "visited"
   */
  FMS_XBAR(bxp)->mrf_resolved = TRUE;
  FMS_XBAR(rxp)->mrf_resolved = TRUE;

  fms_notify(FMS_EVENT_DEBUG, "replacing %s", bxp->linecard->enclosure->name);


  /* for each connected link, disconnect the bogus link and make
   * a new link from the new xbar.
   */
  for (bp=0; bp<bxp->num_ports; ++bp) {

    rp = bp - dp;  /* the real xbar port = bogus xbar port - delta */

    /* get pointers to the "other" nodes and the ports they connect back to
     * rxp and bxp with
     */
    obnp = bxp->topo_ports[bp];
    if (obnp == NULL) continue;

    /* make sure rp is legit */
    if (rp < 0 || rp >= rxp->num_ports) {
      fms_notify(FMS_EVENT_INFO, "Impossible rp (%d), got wrong port", rp);
      return;
    }

    obp = bxp->topo_rports[bp];
    ornp = rxp->topo_ports[rp];
    orp = rxp->topo_rports[rp];

    /* remove the topo link to the other xbar (we'll reconnect it to the 
     * real xbar)
     */
    fms_remove_topo_link(LF_NODE(bxp), bp);

    /* Go no further if we've already visited the bogus xbars' mate */
    if (obnp->ln_type == LF_NODE_XBAR
        && FMS_XBAR(LF_XBAR(obnp))->mrf_resolved
        && FMS_ENC(LF_XBAR(obnp)->linecard->enclosure)->is_bogus) {
      continue;
    }

    /* All bogus physical ports have tranceivers because at the time
     * they were generated it was not know whether they or not they
     * would be needed. Fix this up now. If the the corresponding real
     * xbar is connected directly to another xbar (with no tranceivers)
     * remove them from the bogus xbar.
     */
    if (bxp->phys_ports[bp]->ln_type == LF_NODE_LC_XCVR) {
      bxcp = LF_XCVR(bxp->phys_ports[bp]);

      /* break the physical connection to the bogus xcvr */
      lf_make_phys_link(LF_NODE(bxcp), bxp->phys_rports[bp], NULL, 0);
      lf_make_phys_link(LF_NODE(bxp), bp, NULL, 0);
    }

    /* Linking to someplace new. Given the node the bogus xbar points (obnp)
     * there are 2 cases:
     * 1. obnp has a corresponding "real" xbar (ornp) - follow links further
     * 2. obnp has no corresponding "real" xbar-  move links to obnp to rxp
     */

    /* case 1 - implicit link- a link to an internal xbar */
    if (ornp != NULL) {

      if (obnp->ln_type != LF_NODE_XBAR)
        LF_ERROR(("resolve inconsistency: non-null real, non-xbar bogus"));
      if (ornp->ln_type != LF_NODE_XBAR)
        LF_ERROR(("resolve inconsistency: non-xbar real, xbar bogus"));

      obxp = LF_XBAR(obnp);
      orxp = LF_XBAR(ornp);

      if (FMS_ENC(obxp->linecard->enclosure)->is_bogus
          && ! FMS_ENC(orxp->linecard->enclosure)->is_bogus) {
        // fms_notify(FMS_EVENT_DEBUG, "bonding from %p:%d to %p:%d",
        //    bxp, bp, obxp, obp);
        bond_xbars(orxp, obxp, obp, orp);
      } else {
	if (F.debug) {
	  fms_notify(FMS_EVENT_DEBUG, "not traversing from %p:%d to %p:%d",
	      bxp, bp, obxp, obp);
	}
      }

    } else {
      /* case 2 - connect these xbars directly to eachother */
      // fms_notify(FMS_EVENT_DEBUG,
      //    "NULL ornp port found - copy links, obnp type: %d", obnp->ln_type);
      fms_connect_topo_nodes(LF_NODE(rxp), rp, obnp, obp);
      fms_add_topo_link_to_db(LF_NODE(rxp), rp);
    }

  }
  /* Remove this enclosure from the fabric */
  fms_notify(FMS_EVENT_DEBUG, "removing enclosure %s",
      bxp->linecard->enclosure->name);
  fms_remove_enclosure(bxp->linecard->enclosure);

  return;
 except:
  fms_perror_exit(1);
}

/*
 * invalid routes have been sent to a bogus xbar and the
 * invalid route counts from the switch have been updated. Look at the counts
 * to determine the location of the bogus xbar. 
 */
static void
fms_find_xbar()
{
  struct lf_fabric *fp;
  struct fms_settings *fsetp;
  struct lf_xbar *bxp;	/* bogus xbar */
  struct lf_xbar *rxp;	/* real xbar */
  int rp;               /* real port */
  int x;
  int xbar_index;
  int host_in_port;

  fp = F.fabvars->fabric;
  fsetp = F.settings;

  /* Calculate invalid route count differences for each port */
  fms_res_calc_inv_rt_diffs();

  /* get results of invalid route search */
  xbar_index = FMS_FABRIC(fp)->inv_rt_xbar_index;
  host_in_port = FMS_FABRIC(fp)->inv_rt_host_in_port;

  fms_notify(FMS_EVENT_DEBUG, "fms_find_xbar, xbar:%d, port:%d",
      xbar_index, host_in_port);

  /* lookup the bogus xbar we send invalid routes on */
  bxp = NULL;
  for (x=0; x<fp->num_xbars; x++) {
    if (fp->xbars[x]->x_topo_index == xbar_index) {
      bxp = fp->xbars[x];
      break;
    }
  }
  if (bxp == NULL) LF_ERROR(("xbar %d returned by FMA not found", xbar_index));

  /* Now, find a matching xbar! */
  rxp = fms_find_xbar_with_inv_rts(&rp);

  if (rxp) {

    /* mark all xbars in fabric as unresolved so we can tell who we have 
     * visited
     */
    for (x=0; x<fp->num_xbars; ++x) {
      FMS_XBAR(fp->xbars[x])->mrf_resolved = FALSE;
    }

    /* bond the real and bogus xbars taking care to use the 
     * delta between the bogus and real xbar ports
     */
    bond_xbars(rxp, bxp, host_in_port, rp);

    /* Now that some xbars have been resolved, either do more or indicate that
     * we are done.
     */
    FMS_FABRIC(fp)->resolve_in_prog = FALSE;
    if (fms_fabric_needs_resolution()) {
      fms_resolve_fabric();
    } else {
      fms_state_fabric_resolution_complete();
    }
  } else {

    /* didn't find xbar, retry or quit if max retries reached */
    if (FMS_FABRIC(fp)->inv_rt_retries < fsetp->resolve_retries) {
      fms_switch_set_res_handler(fms_send_inv_rts);

    } else {
      /* Give up resolving xbars for now. Try again next time a topo map is 
       * pushed out (due to linecards or FMAs coming online).
       */
      fms_notify(FMS_EVENT_INFO,
          "Max invalid route retries reached. Giving up xbar resolution.");
      
      /* record the number of bogus xbars remaining so that if/when more come
       * online, we'll know to try again
       */
      FMS_FABRIC(fp)->num_bogus_xbars = fms_count_bogus_xbars();
      fms_state_fabric_resolution_complete();
    }
  }

  return;
 except:
  fms_perror_exit(1);
}

/*
 * find xbar/port on switches with right number of invalid routes 
 */
static struct lf_xbar *
fms_find_xbar_with_inv_rts(int *port)
{
  struct lf_fabric *fp;
  struct fms_settings *fsetp;
  struct lf_xbar *rxp;
  int rp;
  int num_found;
  int num_too_high;
  int e;

  fp = F.fabvars->fabric;
  fsetp = F.settings;

  rxp = NULL;
  rp = 0;
  num_found = 0;
  num_too_high = 0;

  /* loop through enclosures to avoid inspecting xbars belonging to
   * bogus enclosures
   */
  for (e=0; e<fp->num_enclosures; ++e) {
    struct lf_enclosure *ep;
    int l;

    ep = fp->enclosures[e];

    if (FMS_ENC(ep)->is_bogus) continue;

    for (l=0; l<ep->num_slots; ++l) {
      struct lf_linecard *lp;
      int x;

      lp = ep->slots[l];
      if (lp == NULL) continue;
 
      for (x=0; x<lp->num_xbars; ++x) {
        struct lf_xbar *xp;
        int p;

        xp = LF_XBAR(lp->xbars[x]);
        for (p=0; p<xp->num_ports; ++p) {
          struct fms_xbar_port *pp;
          int inv_rt;
          pp = FMS_XBAR(xp)->x_port_data[p];

          /* skip disabled ports */
          /* XXX - if (xp->data[p].cur_vals.control != 0) continue; */

	  inv_rt = pp->res_inv_rt_diff;

          if (F.debug && inv_rt > fsetp->resolve_packet_min - 100) {
            fms_notify(FMS_EVENT_DEBUG, "%s:%d:%d:%d %d inv rts", 
                xp->linecard->enclosure->name, l, x, p, inv_rt);
          }

          if (inv_rt > fsetp->resolve_packet_max) num_too_high++;

          if (inv_rt >= fsetp->resolve_packet_min
	      && inv_rt <= fsetp->resolve_packet_max) {
            num_found++; 
            rxp = xp;
            rp = p;
          }
        }
      }
    }
  }

  if (num_found == 1 && num_too_high == 0) {
    *port = rp;
    return(rxp);

  } else {
    if (num_found == 0) { 
      fms_notify(FMS_EVENT_DEBUG, "Couldn't find xbar via invalid routes");
    } else {
      fms_notify(FMS_EVENT_DEBUG, "Too many xbar invalid route matches");
    }
    return NULL;
  }
}
